#!/bin/bash

# Function to display the dashboard
show_dashboard() {
    clear
    echo "Overspace Management Script"
    echo "1. Scan for overspace files"
    echo "2. Copy overspace files to backup server"
    echo "3. Remove overspace files"
    echo "4. File validator / Checker"
    echo "5. Exit"
    echo
    read -p "Enter your choice (1-4): " choice
    echo
}

# Function to scan for overspace files
scan_overspace() {
    echo "Scanning for files larger than 5GB..."
    read -p "Enter the directory to scan (default: /home): " scan_dir
    scan_dir=${scan_dir:-/home}
    read -p "Enter the output file path (default: overspace_users.txt): " output_file
    output_file=${output_file:-overspace_users.txt}

    echo "Scanning $scan_dir for files larger than 5GB. Results will be saved in $output_file"
    find "$scan_dir" -type f -size +5G > "$output_file"
    echo "Scan complete. Results saved in $output_file"
    echo "Found $(wc -l < "$output_file") files."
    read -p "Press Enter to continue..."
}

# Function to copy overspace files
copy_overspace() {
    read -p "Enter the path to the overspace_users.txt file: " input_file
    if [ ! -f "$input_file" ]; then
        echo "File $input_file not found. Please run the scan first or provide a valid file."
        read -p "Press Enter to continue..."
        return
    fi

    read -p "Enter the destination server (e.g., user@server): " dest_server
    read -p "Enter the SSH port number (default: 22): " ssh_port
    ssh_port=${ssh_port:-22}
    read -p "Enter the base backup directory on the destination server: " backup_dir
    
    read -p "Do you want to use an identity file for SSH authentication? (y/n): " use_identity
    if [ "$use_identity" = "y" ] || [ "$use_identity" = "Y" ]; then
        read -p "Enter the path to the identity file: " identity_file
        if [ ! -f "$identity_file" ]; then
            echo "Identity file not found. Proceeding without it."
            ssh_options="-p $ssh_port"
        else
            ssh_options="-p $ssh_port -i $identity_file"
        fi
    else
        ssh_options="-p $ssh_port"
    fi
    
    echo "Copying files to $dest_server:$backup_dir using port $ssh_port"
    if [ -n "$identity_file" ] && [ -f "$identity_file" ]; then
        echo "Using identity file: $identity_file"
    fi

    while IFS= read -r file; do
        echo "Copying $file..."
        # Extract the relative path
        relative_path=${file#/home/}
        # Create the directory structure on the remote server
        ssh $ssh_options $dest_server "mkdir -p $backup_dir/$(dirname $relative_path)"
        # Copy the file preserving the relative path
        rsync -avz --progress -e "ssh $ssh_options" "$file" "${dest_server}:${backup_dir}/${relative_path}"
    done < "$input_file"

    echo "Copy complete."
    read -p "Press Enter to continue..."
}

# Function to remove overspace files
remove_overspace() {
    read -p "Enter the path to the overspace_users.txt file (default: overspace_users.txt): " input_file
    input_file=${input_file:-overspace_users.txt}

    if [ ! -f "$input_file" ]; then
        echo "File $input_file not found. Please run the scan first or provide a valid file."
        read -p "Press Enter to continue..."
        return
    fi

    echo "The following files will be moved to trash:"
    cat "$input_file"
    echo

    read -p "Are you sure you want to move these files to trash? (y/n): " confirm

    if [ "$confirm" = "y" ] || [ "$confirm" = "Y" ]; then
        echo "Moving overspace files to trash..."
        while IFS= read -r file; do
            echo "Trashing $file"
            trash "$file"
        done < "$input_file"
        echo "Removal complete. Files have been moved to trash."
    else
        echo "Operation cancelled."
    fi

    read -p "Press Enter to continue..."
}


# Function to check file size and location
check_file() {
    read -p "Enter the path to the overspace_users.txt file (default: overspace_users.txt): " input_file
    input_file=${input_file:-overspace_users.txt}

    if [ ! -f "$input_file" ]; then
        echo "File $input_file not found. Please run the scan first or provide a valid file."
        read -p "Press Enter to continue..."
        return
    fi

    echo "Checking files listed in $input_file:"
    echo "--------------------------------------"

    while IFS= read -r file_path; do
        if [ -f "$file_path" ]; then
            size=$(du -sh "$file_path" | cut -f1)
            location=$(dirname "$file_path")
            echo "File: $file_path"
            echo "Size: $size"
            echo "Location: $location"
        else
            echo "File not found: $file_path"
        fi
        echo "--------------------------------------"
    done < "$input_file"

    read -p "Press Enter to continue..."
}

# Main loop
while true; do
    show_dashboard

    case $choice in
        1) scan_overspace ;;
        2) copy_overspace ;;
        3) remove_overspace ;;
        4) check_file ;;
        5) echo "Exiting..."; exit 0 ;;
        *) echo "Invalid choice. Please try again."; read -p "Press Enter to continue..." ;;
    esac
done